
<?php
// Retrieve existing todo items from JSON file
$todos = array();
if (file_exists('todos.json')) {
    $todos = json_decode(file_get_contents('todos.json'), true);
} else {
    // Debugging: If the file doesn't exist, log a message
    error_log('todos.json file not found');
}

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'];

    switch ($action) {
        case 'create':
            $todo = array(
                'id' => uniqid(),
                'text' => $_POST['text'],
                'completed' => false
            );
            $todos[] = $todo;
            file_put_contents('todos.json', json_encode($todos));
            echo json_encode($todo);
            break;

        case 'update':
            $todoId = $_POST['id'];
            $text = $_POST['text'];
            $completed = $_POST['completed'];
            $found = false;
            foreach ($todos as &$todo) {
                if ($todo['id'] === $todoId) {
                    $todo['text'] = $text;
                    $todo['completed'] = $completed;
                    $found = true;
                    break;
                }
            }
            if (!$found) {
                // Debugging: Log a message if the todo item is not found
                error_log("Todo item with ID $todoId not found");
            }
            file_put_contents('todos.json', json_encode($todos));
            echo json_encode($todos);
            break;

        case 'delete':
            $todoId = $_POST['id'];
            $todos = array_filter($todos, function ($todo) use ($todoId) {
                return $todo['id'] !== $todoId;
            });
            file_put_contents('todos.json', json_encode($todos));
            echo json_encode($todos);
            break;
    }
}
?>

<!DOCTYPE html>
<html>
<head>
    <title>Todo List</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <style>
        .completed {
            text-decoration: line-through;
        }
    </style>
</head>
<body>
    <h1>Todo List</h1>
    <form id="todo-form">
        <input type="text" id="todo-input" placeholder="Enter a new todo">
        <button type="submit">Add</button>
    </form>
    <ul id="todo-list"></ul>

    <script>
        // Load todos from the server
        const loadTodos = () => {
            fetch('', {
                method: 'GET'
            })
            .then(response => response.json())
            .then(data => {
                const todoList = document.getElementById('todo-list');
                todoList.innerHTML = '';
                data.forEach(todo => {
                    const li = document.createElement('li');
                    li.textContent = todo.text;
                    if (todo.completed) {
                        li.classList.add('completed');
                    }
                    const editBtn = document.createElement('button');
                    editBtn.textContent = 'Edit';
                    editBtn.addEventListener('click', () => editTodo(todo.id, todo.text, todo.completed));
                    const deleteBtn = document.createElement('button');
                    deleteBtn.textContent = 'Delete';
                    deleteBtn.addEventListener('click', () => deleteTodo(todo.id));
                    li.appendChild(editBtn);
                    li.appendChild(deleteBtn);
                    todoList.appendChild(li);
                });

                // Debugging: Log the length of the todo list
                console.log(`Loaded ${data.length} todo items`);
            })
            .catch(error => console.error('Error:', error));
        };

        // Create a new todo
        const createTodo = (text) => {
            fetch('', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded'
                },
                body: `action=create&text=${encodeURIComponent(text)}`
            })
            .then(response => response.json())
            .then(data => {
                loadTodos();
            })
            .catch(error => console.error('Error:', error));
        };

        // Edit an existing todo
        const editTodo = (id, text, completed) => {
            const newText = prompt('Edit todo:', text);
            if (newText !== null) {
                const newCompleted = confirm('Mark as completed?');
                fetch('', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded'
                    },
                    body: `action=update&id=${id}&text=${encodeURIComponent(newText)}&completed=${newCompleted}`
                })
                .then(response => response.json())
                .then(data => {
                    loadTodos();
                })
                .catch(error => console.error('Error:', error));
            }
        };

        // Delete a todo
        const deleteTodo = (id) => {
            if (confirm('Are you sure you want to delete this todo?')) {
                fetch('', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded'
                    },
                    body: `action=delete&id=${id}`
                })
                .then(response => response.json())
                .then(data => {
                    loadTodos();
                })
                .catch(error => console.error('Error:', error));
            }
        };

        // Add event listener to the form
        const todoForm = document.getElementById('todo-form');
        todoForm.addEventListener('submit', (event) => {
            event.preventDefault();
            const todoInput = document.getElementById('todo-input');
            const text = todoInput.value.trim();
            if (text !== '') {
                createTodo(text);
                todoInput.value = '';
            }
        });

        // Load initial todos
        loadTodos();
    </script>
</body>
</html>
```