<?php 
$projectsDir = 'projects/';

if (isset($_GET['project'])) {
    $projectName = $_GET['project'];
    $projectPath = $projectsDir . $projectName;
    $filePath = $projectPath . '/index.php';
    $configPath = $projectPath . '/config.json';
    $versionsPath = $projectPath . '/versions.json';

    if (isset($_GET['file'])) {
        $selectedFile = $_GET['file'];
        $filePath = $projectPath . '/' . $selectedFile;
        if (file_exists($filePath)) {
            $fileContents = file_get_contents($filePath);
        } else {
            $message = "Selected file not found.";
        }
    }

    if (file_exists($filePath)) {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (isset($_POST['file_contents'])) {
                $oldContents = file_get_contents($filePath);
                $newContents = $_POST['file_contents'];
                if ($oldContents !== $newContents) {
                    file_put_contents($filePath, $newContents);
                    $aiRequest = $_POST['ai_request'] ?? "Manual save";
                    saveVersion($versionsPath, $selectedFile, $oldContents, $newContents, $aiRequest, "");
                    $message = "File saved successfully and version recorded.";
                } else {
                    $message = "No changes detected. File not saved.";
                }
            }
            if (isset($_POST['languages'])) {
                $config = ['languages' => $_POST['languages']];
                file_put_contents($configPath, json_encode($config));
                $message = "Languages saved successfully.";
            }
        }
        $fileContents = file_get_contents($filePath);
        $config = file_exists($configPath) ? json_decode(file_get_contents($configPath), true) : ['languages' => []];
    } else {
        $message = "Project file not found.";
    }
} else {
    $message = "No project specified.";
}

function getApiKey() {
    $credentialsFile = 'credentials.json';
    if (file_exists($credentialsFile)) {
        $credentials = json_decode(file_get_contents($credentialsFile), true);
        return $credentials['apiKey'] ?? null;
    }
    return null;
}

function callClaudeAPI($request, $fileContent, $language) {
    $apiUrl = 'https://api.anthropic.com/v1/messages';
    $apiKey = getApiKey();

    if (!$apiKey) {
        return ['error' => 'API key not set. Please set your Claude API key.'];
    }

    $data = [
        'model' => 'claude-3-sonnet-20240229',
        'max_tokens' => 4000,
        'messages' => [
            ['role' => 'user', 'content' => "Here's the current code for the entire page:\n\n```$language\n$fileContent\n```\n\nBased on this existing code, please provide only $language code for the following request:\n\n$request\n\nRespond with only the $language code, no explanations or markdown formatting."]
        ]
    ];

    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'x-api-key: ' . $apiKey,
        'anthropic-version: 2023-06-01'
    ]);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

    curl_close($ch);

    $result = json_decode($response, true);

    if ($httpCode != 200) {
        return [
            'error' => 'API request failed: ' . ($result['error']['message'] ?? 'Unknown error'),
            'http_code' => $httpCode,
            'response' => $response
        ];
    }

    return $result;
}

$apiTestResult = '';
$debugInfo = '';
$requestToSend = '';

if (isset($_POST['action']) && $_POST['action'] == 'preview') {
    $aiRequest = $_POST['ai_request'];
    $fileContents = $_POST['file_contents'];
    $languages = $_POST['languages'];
    
    if (empty($languages)) {
        echo json_encode(['error' => 'Please select at least one language']);
    } else {
        $requestToSend = "Here's the current code:\n\n```\n$fileContents\n```\n\nBased on this existing code, please provide code for the following request using these languages: " . implode(', ', $languages) . "\n\n$aiRequest\n\nRespond with only the code, make sure to supply the whole code to easily copy paste, no explanations or markdown formatting.";
        echo json_encode(['preview' => $requestToSend]);
    }
    exit;
}

function saveVersion($versionsPath, $fileName, $oldCode, $newCode, $request, $debug) {
    $versions = [];
    if (file_exists($versionsPath)) {
        $versions = json_decode(file_get_contents($versionsPath), true);
    }
    
    $versions[] = [
        'timestamp' => date('Y-m-d H:i:s'),
        'file' => $fileName,
        'oldCode' => $oldCode,
        'newCode' => $newCode,
        'request' => $request,
        'debug' => $debug
    ];
    
    file_put_contents($versionsPath, json_encode($versions, JSON_PRETTY_PRINT));
}

if (isset($_POST['action']) && $_POST['action'] == 'generate_code') {
    $aiRequest = $_POST['ai_request'];
    $languages = $_POST['languages'];
    $languagesStr = implode(', ', $languages);
    $aiResponse = callClaudeAPI($aiRequest, $fileContents, $languagesStr);
    
    if (!isset($aiResponse['error'])) {
        $generatedCode = $aiResponse['content'][0]['text'] ?? '';
        echo json_encode([
            'success' => true, 
            'generatedCode' => $generatedCode,
            'debug' => json_encode($aiResponse, JSON_PRETTY_PRINT)
        ]);
    } else {
        echo json_encode(['success' => false, 'error' => $aiResponse['error']]);
    }
    exit;
}

if (isset($_POST['action']) && $_POST['action'] == 'create_file') {
    $fileName = $_POST['file_name'];
    $newFilePath = $projectPath . '/' . $fileName;
    
    if (file_exists($newFilePath)) {
        echo json_encode(['success' => false, 'error' => 'File already exists']);
    } else {
        if (file_put_contents($newFilePath, '') !== false) {
            echo json_encode(['success' => true]);
        } else {
            echo json_encode(['success' => false, 'error' => 'Failed to create file']);
        }
    }
    exit;
}

$availableLanguages = [
    'php' => 'PHP',
    'javascript' => 'JavaScript',
    'html' => 'HTML',
    'css' => 'CSS',
    'python' => 'Python',
    'java' => 'Java',
    'csharp' => 'C#',
    'cpp' => 'C++',
    'ruby' => 'Ruby',
    'go' => 'Go'
];

// Add this new function to load todo.php content
function loadTodoContent($projectName) {
    $todoPath = "todo.php";
    if (file_exists($todoPath)) {
        ob_start();
        include $todoPath;
        return ob_get_clean();
    }
    return "<p>Todo list not found for this project.</p>";
}

$todoContent = loadTodoContent($projectName);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Project: <?php echo htmlspecialchars($projectName); ?></title>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            padding: 20px;
            max-width: 1200px;
            margin: 0 auto;
        }
        .tab {
            overflow: hidden;
            border: 1px solid #ccc;
            background-color: #f1f1f1;
        }
        .tab button {
            background-color: inherit;
            float: left;
            border: none;
            outline: none;
            cursor: pointer;
            padding: 14px 16px;
            transition: 0.3s;
        }
        .tab button:hover {
            background-color: #ddd;
        }
        .tab button.active {
            background-color: #ccc;
        }
        .tabcontent {
            display: none;
            padding: 6px 12px;
            border: 1px solid #ccc;
            border-top: none;
        }
        textarea, select, input[type="text"] {
            width: 100%;
            max-width: 100%;
            box-sizing: border-box;
        }
        #fileList {
            width: auto;
            display: inline-block;
        }
        .spinner {
            display: none;
            /* Add your spinner styling here */
        }
        .language-checkboxes {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
        }
        .language-checkboxes label {
            display: flex;
            align-items: center;
        }
        .ai-tabs {
            display: flex;
            border-bottom: 1px solid #ccc;
            margin-bottom: 10px;
        }
        .ai-tab {
            padding: 10px 15px;
            cursor: pointer;
            background-color: #f1f1f1;
            border: 1px solid #ccc;
            border-bottom: none;
            margin-right: 5px;
        }
        .ai-tab.active {
            background-color: #fff;
            border-bottom: 1px solid #fff;
            margin-bottom: -1px;
        }
        .ai-content {
            display: none;
            padding: 15px;
            border: 1px solid #ccc;
            border-top: none;
        }
        .ai-content.active {
            display: block;
        }
        .spinner {
            border: 4px solid #f3f3f3;
            border-top: 4px solid #3498db;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            animation: spin 1s linear infinite;
            margin: 20px auto;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <h1>Edit Project: <?php echo htmlspecialchars($projectName); ?></h1>

    <div class="tab">
        <button class="tablinks active" onclick="openTab(event, 'Editor')">Editor</button>
        <button class="tablinks" onclick="openTab(event, 'Todo')">Todo</button>
    </div>

    <div id="Editor" class="tabcontent" style="display: block;">
        <p><a href="versions.php?project=<?php echo urlencode($projectName); ?>&file=<?php echo urlencode($selectedFile); ?>">View Version History</a></p>

        <div style="margin-bottom: 10px;">
            <select id="fileList">
                <option value="">Select a file</option>
                <?php
                $files = glob($projectPath . '/*');
                foreach ($files as $file) {
                    $fileName = basename($file);
                    echo "<option value=\"" . htmlspecialchars($fileName) . "\">" . htmlspecialchars($fileName) . "</option>";
                }
                ?>
            </select>
            <a href="#" id="newFileLink">Create New File</a>
        </div>

        <?php if (isset($message)): ?>
            <p><?php echo $message; ?></p>
        <?php endif; ?>

        <form method="POST" action="" id="save-form">
            <textarea name="file_contents" id="file_contents" rows="20"><?php echo htmlspecialchars($fileContents); ?></textarea>
            <br>
            <input type="hidden" name="ai_request" id="hidden_ai_request" value="">
            <button type="submit">Save Changes</button>
        </form>

        <h3>Project Languages</h3>
        <form method="POST" action="">
            <div class="language-checkboxes">
                <?php foreach ($availableLanguages as $value => $label): ?>
                    <label>
                        <input type="checkbox" name="languages[]" value="<?php echo $value; ?>"
                            <?php echo in_array($value, $config['languages']) ? 'checked' : ''; ?>>
                        <?php echo $label; ?>
                    </label>
                <?php endforeach; ?>
            </div>
            <br>
            <button type="submit">Save Languages</button>
        </form>

        <p><a href="index.php">Back to Project List</a></p>
        <p><a href="<?php echo htmlspecialchars($projectsDir . $projectName); ?>" target="_blank">View Project Page</a></p>
        
        <div class="ai-tabs">
            <div class="ai-tab active" data-tab="ask-ai">Ask AI</div>
            <div class="ai-tab" data-tab="preview-request">Preview Request</div>
            <div class="ai-tab" data-tab="generated-code">Generated Code</div>
        </div>

        <div id="ask-ai" class="ai-content active">
            <h3>Ask Claude AI to write code</h3>
            <textarea id="ai_request" rows="5" placeholder="Enter your request for Claude AI"><?php echo htmlspecialchars($_POST['ai_request'] ?? ''); ?></textarea>
            <br>
            <button type="button" id="preview_request">Preview Request</button>
        </div>

        <div id="preview-request" class="ai-content">
            <h3>Request to be sent:</h3>
            <textarea id="request_preview" rows="10" readonly></textarea>
            <br>
            <button type="button" id="generate_code">Generate Code</button>
        </div>

        <div id="generated-code" class="ai-content">
            <h3>Generated Code:</h3>
            <textarea id="generated_code" rows="20" readonly></textarea>
            <br>
            <button type="button" id="accept_generated_code">Accept Generated Code</button>
        </div>

        <div class="spinner" id="loading-spinner" style="display: none;"></div>

        <h3>Debug Information:</h3>
        <textarea id="debug_info" rows="10" readonly></textarea>
    </div>

    <div id="Todo" class="tabcontent">
        <?php echo $todoContent; ?>
    </div>

    <script>
    function openTab(evt, tabName) {
        var i, tabcontent, tablinks;
        tabcontent = document.getElementsByClassName("tabcontent");
        for (i = 0; i < tabcontent.length; i++) {
            tabcontent[i].style.display = "none";
        }
        tablinks = document.getElementsByClassName("tablinks");
        for (i = 0; i < tablinks.length; i++) {
            tablinks[i].className = tablinks[i].className.replace(" active", "");
        }
        document.getElementById(tabName).style.display = "block";
        evt.currentTarget.className += " active";
    }

    function switchAiTab(tabName) {
        $('.ai-tab').removeClass('active');
        $('.ai-content').removeClass('active');
        $(`.ai-tab[data-tab="${tabName}"]`).addClass('active');
        $(`#${tabName}`).addClass('active');
    }

    $(document).ready(function() {
        $('.ai-tab').click(function() {
            switchAiTab($(this).data('tab'));
        });

        $('#preview_request').click(function() {
            var aiRequest = $('#ai_request').val();
            var fileContents = $('#file_contents').val();
            var languages = $('input[name="languages[]"]:checked').map(function() {
                return this.value;
            }).get();
            
            $('#loading-spinner').show();
            
            $.ajax({
                url: '',
                method: 'POST',
                data: {
                    action: 'preview',
                    ai_request: aiRequest,
                    file_contents: fileContents,
                    languages: languages
                },
                dataType: 'json',
                success: function(response) {
                    $('#loading-spinner').hide();
                    if (response.error) {
                        $('#debug_info').val("Error: " + response.error);
                        $('#request_preview').val('');
                    } else {
                        $('#request_preview').val(response.preview);
                        $('#debug_info').val('');
                        switchAiTab('preview-request');
                    }
                },
                error: function(jqXHR, textStatus, errorThrown) {
                    $('#loading-spinner').hide();
                    $('#debug_info').val("Error generating preview: " + textStatus + "\n" + errorThrown);
                    $('#request_preview').val('');
                }
            });
        });

        $('#generate_code').click(function() {
            var aiRequest = $('#ai_request').val();
            var fileContents = $('#file_contents').val();
            var languages = $('input[name="languages[]"]:checked').map(function() {
                return this.value;
            }).get();
            
            $('#loading-spinner').show();
            
            $.ajax({
                url: '',
                method: 'POST',
                data: {
                    action: 'generate_code',
                    ai_request: aiRequest,
                    file_contents: fileContents,
                    languages: languages
                },
                dataType: 'json',
                success: function(response) {
                    $('#loading-spinner').hide();
                    if (response.success) {
                        $('#generated_code').val(response.generatedCode);
                        $('#debug_info').val(response.debug);
                        switchAiTab('generated-code');
                    } else {
                        $('#debug_info').val("Error: " + response.error);
                    }
                },
                error: function() {
                    $('#loading-spinner').hide();
                    $('#debug_info').val("Error generating code");
                }
            });
        });

        $('#accept_generated_code').click(function() {
            var generatedCode = $('#generated_code').val();
            $('#file_contents').val(generatedCode);
            $('#hidden_ai_request').val($('#ai_request').val());
            switchAiTab('ask-ai');
        });

        $('#save-form').submit(function() {
            var aiRequest = $('#ai_request').val();
            $('#hidden_ai_request').val(aiRequest || 'Manual save');
        });

        $('#fileList').change(function() {
            var selectedFile = $(this).val();
            if (selectedFile) {
                window.location.href = '?project=<?php echo urlencode($projectName); ?>&file=' + encodeURIComponent(selectedFile);
            }
        });

        $('#newFileLink').click(function(e) {
            e.preventDefault();
            var fileName = prompt('Enter the new file name:');
            if (fileName) {
                $.ajax({
                    url: '',
                    method: 'POST',
                    data: {
                        action: 'create_file',
                        project: '<?php echo $projectName; ?>',
                        file_name: fileName
                    },
                    dataType: 'json',
                    success: function(response) {
                        if (response.success) {
                            alert('File created successfully!');
                            location.reload();
                        } else {
                            alert('Failed to create file: ' + response.error);
                        }
                    },
                    error: function() {
                        alert('Error creating file');
                    }
                });
            }
        });
    });
    </script>
</body>
</html>
