<?php
$projectsDir = 'projects/';

if (isset($_GET['project'])) {
    $projectName = $_GET['project'];
    $projectPath = $projectsDir . $projectName;
    $filePath = $projectPath . '/index.php';
    $configPath = $projectPath . '/config.json';

    if (isset($_GET['file'])) {
        $selectedFile = $_GET['file'];
        $filePath = $projectPath . '/' . $selectedFile;
        if (file_exists($filePath)) {
            $fileContents = file_get_contents($filePath);
        } else {
            $message = "Selected file not found.";
        }
    }

    if (file_exists($filePath)) {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (isset($_POST['file_contents'])) {
                $fileContents = $_POST['file_contents'];
                file_put_contents($filePath, $fileContents);
                $message = "File saved successfully.";
            }
            if (isset($_POST['languages'])) {
                $config = ['languages' => $_POST['languages']];
                file_put_contents($configPath, json_encode($config));
                $message = "Languages saved successfully.";
            }
        }
        $fileContents = file_get_contents($filePath);
        $config = file_exists($configPath) ? json_decode(file_get_contents($configPath), true) : ['languages' => []];
    } else {
        $message = "Project file not found.";
    }
} else {
    $message = "No project specified.";
}

function getApiKey() {
    $credentialsFile = 'credentials.json';
    if (file_exists($credentialsFile)) {
        $credentials = json_decode(file_get_contents($credentialsFile), true);
        return $credentials['apiKey'] ?? null;
    }
    return null;
}

function callClaudeAPI($request, $fileContent, $language) {
    $apiUrl = 'https://api.anthropic.com/v1/messages';
    $apiKey = getApiKey();

    if (!$apiKey) {
        return ['error' => 'API key not set. Please set your Claude API key.'];
    }

    $data = [
        'model' => 'claude-3-sonnet-20240229',
        'max_tokens' => 1000,
        'messages' => [
            ['role' => 'user', 'content' => "Here's the current code for the entire page:\n\n```$language\n$fileContent\n```\n\nBased on this existing code, please provide only $language code for the following request:\n\n$request\n\nRespond with only the $language code, no explanations or markdown formatting."]
        ]
    ];

    $ch = curl_init($apiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'x-api-key: ' . $apiKey,
        'anthropic-version: 2023-06-01'
    ]);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

    curl_close($ch);

    $result = json_decode($response, true);

    if ($httpCode != 200) {
        return [
            'error' => 'API request failed: ' . ($result['error']['message'] ?? 'Unknown error'),
            'http_code' => $httpCode,
            'response' => $response
        ];
    }

    return $result;
}

$apiTestResult = '';
$debugInfo = '';
$requestToSend = '';

// Add this new endpoint for AJAX preview
if (isset($_POST['action']) && $_POST['action'] == 'preview') {
    $aiRequest = $_POST['ai_request'];
    $fileContents = $_POST['file_contents'];
    $languages = $_POST['languages'];
    
    if (empty($languages)) {
        echo json_encode(['error' => 'Please select at least one language']);
    } else {
        $requestToSend = "Here's the current code:\n\n```\n$fileContents\n```\n\nBased on this existing code, please provide code for the following request using these languages: " . implode(', ', $languages) . "\n\n$aiRequest\n\nRespond with only the code, no explanations or markdown formatting.";
        echo json_encode(['preview' => $requestToSend]);
    }
    exit;
}

// Modify the generate code endpoint to return JSON
if (isset($_POST['action']) && $_POST['action'] == 'generate_code') {
    $aiRequest = $_POST['ai_request'];
    $languages = $_POST['languages'];
    $languagesStr = implode(', ', $languages);
    $aiResponse = callClaudeAPI($aiRequest, $fileContents, $languagesStr);
    
    if (!isset($aiResponse['error'])) {
        $generatedCode = $aiResponse['content'][0]['text'] ?? '';
        $fileContents .= "\n\n" . $generatedCode;
        echo json_encode(['success' => true, 'generatedCode' => $generatedCode]);
    } else {
        echo json_encode(['success' => false, 'error' => $aiResponse['error']]);
    }
    exit;
}

// Add this new endpoint for file creation
if (isset($_POST['action']) && $_POST['action'] == 'create_file') {
    $fileName = $_POST['file_name'];
    $newFilePath = $projectPath . '/' . $fileName;
    
    if (file_exists($newFilePath)) {
        echo json_encode(['success' => false, 'error' => 'File already exists']);
    } else {
        if (file_put_contents($newFilePath, '') !== false) {
            echo json_encode(['success' => true]);
        } else {
            echo json_encode(['success' => false, 'error' => 'Failed to create file']);
        }
    }
    exit;
}

$availableLanguages = [
    'php' => 'PHP',
    'javascript' => 'JavaScript',
    'html' => 'HTML',
    'css' => 'CSS',
    'python' => 'Python',
    'java' => 'Java',
    'csharp' => 'C#',
    'cpp' => 'C++',
    'ruby' => 'Ruby',
    'go' => 'Go'
];
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Edit Project: <?php echo htmlspecialchars($projectName); ?></title>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <style>
        .spinner {
            border: 4px solid #f3f3f3;
            border-top: 4px solid #3498db;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            display: none;
            margin: 20px auto;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        #fileList {
            margin-right: 10px;
        }
    </style>
</head>
<body>
    <h1>Edit Project: <?php echo htmlspecialchars($projectName); ?></h1>

    <select id="fileList">
        <option value="">Select a file</option>
        <?php
        $files = glob($projectPath . '/*');
        foreach ($files as $file) {
            $fileName = basename($file);
            echo "<option value=\"" . htmlspecialchars($fileName) . "\">" . htmlspecialchars($fileName) . "</option>";
        }
        ?>
    </select>
    <a href="#" id="newFileLink">Create New File</a>

    <?php if (isset($message)): ?>
        <p><?php echo $message; ?></p>
    <?php endif; ?>

    <form method="POST" action="">
        <textarea name="file_contents" id="file_contents" rows="20" cols="80"><?php echo htmlspecialchars($fileContents); ?></textarea>
        <br>
        <button type="submit">Save Changes</button>
    </form>

    <h3>Project Languages</h3>
    <form method="POST" action="">
        <?php foreach ($availableLanguages as $value => $label): ?>
            <label>
                <input type="checkbox" name="languages[]" value="<?php echo $value; ?>"
                    <?php echo in_array($value, $config['languages']) ? 'checked' : ''; ?>>
                <?php echo $label; ?>
            </label>
        <?php endforeach; ?>
        <br>
        <button type="submit">Save Languages</button>
    </form>

    <p><a href="index.php">Back to Project List</a></p>
    <p><a href="<?php echo htmlspecialchars($projectsDir . $projectName); ?>" target="_blank">View Project Page</a></p>
    
    <h3>Ask Claude AI to write code</h3>
    <form id="ai-form" method="POST" action="">
        <textarea name="ai_request" id="ai_request" rows="5" cols="80" placeholder="Enter your request for Claude AI"><?php echo htmlspecialchars($_POST['ai_request'] ?? ''); ?></textarea>
        <br>
        <button type="button" id="preview_request">Preview Request</button>
        <button type="button" id="generate_code">Generate Code</button>
    </form>

    <div class="spinner" id="loading-spinner"></div>

    <h3>Request to be sent:</h3>
    <textarea id="request_preview" rows="10" cols="80" readonly></textarea>

    <h3>Generated Code:</h3>
    <textarea id="generated_code" rows="20" cols="80" readonly></textarea>

    <h3>Debug Information:</h3>
    <textarea id="debug_info" rows="10" cols="80" readonly></textarea>

    <script>
    $(document).ready(function() {
        $('#preview_request').click(function() {
            var aiRequest = $('#ai_request').val();
            var fileContents = $('#file_contents').val();
            var languages = $('input[name="languages[]"]:checked').map(function() {
                return this.value;
            }).get();
            
            $.ajax({
                url: '',
                method: 'POST',
                data: {
                    action: 'preview',
                    ai_request: aiRequest,
                    file_contents: fileContents,
                    languages: languages
                },
                dataType: 'json',
                success: function(response) {
                    if (response.error) {
                        $('#debug_info').val("Error: " + response.error);
                        $('#request_preview').val('');
                    } else {
                        $('#request_preview').val(response.preview);
                        $('#debug_info').val('');
                    }
                },
                error: function(jqXHR, textStatus, errorThrown) {
                    $('#debug_info').val("Error generating preview: " + textStatus + "\n" + errorThrown);
                    $('#request_preview').val('');
                }
            });
        });

        $('#generate_code').click(function() {
            var aiRequest = $('#ai_request').val();
            var fileContents = $('#file_contents').val();
            var languages = $('input[name="languages[]"]:checked').map(function() {
                return this.value;
            }).get();
            
            $('#loading-spinner').show();
            
            $.ajax({
                url: '',
                method: 'POST',
                data: {
                    action: 'generate_code',
                    ai_request: aiRequest,
                    file_contents: fileContents,
                    languages: languages
                },
                dataType: 'json',
                success: function(response) {
                    $('#loading-spinner').hide();
                    if (response.success) {
                        $('#generated_code').val(response.generatedCode);
                        $('#file_contents').val(response.generatedCode); // Replace the content instead of appending
                    } else {
                        $('#debug_info').val("Error: " + response.error);
                    }
                },
                error: function() {
                    $('#loading-spinner').hide();
                    alert('Error generating code');
                }
            });
        });

        $('#fileList').change(function() {
            var selectedFile = $(this).val();
            if (selectedFile) {
                window.location.href = '?project=<?php echo urlencode($projectName); ?>&file=' + encodeURIComponent(selectedFile);
            }
        });

        $('#newFileLink').click(function(e) {
            e.preventDefault();
            var fileName = prompt('Enter the new file name:');
            if (fileName) {
                $.ajax({
                    url: '',
                    method: 'POST',
                    data: {
                        action: 'create_file',
                        project: '<?php echo $projectName; ?>',
                        file_name: fileName
                    },
                    dataType: 'json',
                    success: function(response) {
                        if (response.success) {
                            alert('File created successfully!');
                            location.reload();
                        } else {
                            alert('Failed to create file: ' + response.error);
                        }
                    },
                    error: function() {
                        alert('Error creating file');
                    }
                });
            }
        });
    });
    </script>
</body>
</html>
