
<?php
// Check if todo.json file exists, if not, create it
if (!file_exists('todo.json')) {
    $initialTasks = [];
    $tasksData = json_encode($initialTasks);
    file_put_contents('todo.json', $tasksData);
}

$tasksData = file_get_contents('todo.json');
$tasks = json_decode($tasksData, true);

// Add unique IDs to tasks that don't have one
$nextId = 1;
foreach ($tasks as &$task) {
    if (!isset($task['id'])) {
        $task['id'] = $nextId++;
    }
}

// Check if the project parameter is set
$project = isset($_GET['project']) ? $_GET['project'] : 'default';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['tasks'])) {
        $tasksData = $_POST['tasks'];
        $tasks = json_decode($tasksData, true);
        $tasksData = json_encode($tasks);
        file_put_contents('todo.json', $tasksData);
        echo "Tasks saved successfully.";
        exit();
    } else {
        $action = $_POST['action'];
        $taskId = $_POST['taskId'];

        switch ($action) {
            case 'complete':
                foreach ($tasks as &$task) {
                    if ($task['id'] == $taskId) {
                        $task['completed'] = true;
                        break;
                    }
                }
                break;
            case 'undo':
                foreach ($tasks as &$task) {
                    if ($task['id'] == $taskId) {
                        $task['completed'] = false;
                        break;
                    }
                }
                break;
        }

        $tasksData = json_encode($tasks);
        file_put_contents('todo.json', $tasksData);
        header('Content-Type: application/json');
        echo json_encode($tasks);
        exit();
    }
}
?>
<!DOCTYPE html>
<html>
<head>
    <title>Todo List</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <style>
        .completed {
            text-decoration: line-through;
        }
        body {
            font-family: Arial, sans-serif;
            background-color: #f4f4f4;
            padding: 20px;
        }
        h1 {
            text-align: center;
            color: #333;
        }
        textarea {
            width: 100%;
            height: 100px;
            padding: 10px;
            box-sizing: border-box;
            font-size: 16px;
            resize: vertical;
        }
        textarea.full-screen {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: white;
            z-index: 1;
            border: none;
            padding: 20px;
            box-sizing: border-box;
        }
        button {
            background-image: linear-gradient(to right, #4285F4, #41B6E6);
            color: white;
            padding: 10px 20px;
            border: none;
            cursor: pointer;
        }
        ul {
            list-style-type: none;
            padding: 0;
        }
        li {
            background-color: white;
            padding: 10px;
            margin-bottom: 10px;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.1);
            display: flex;
            flex-direction: column;
            cursor: move;
            position: relative;
        }
        li span {
            margin-bottom: 10px;
        }
        li .button-container {
            display: flex;
            justify-content: flex-end;
            gap: 10px;
        }
        .move-icon {
            position: absolute;
            top: 5px;
            right: 5px;
            font-size: 16px;
            color: #888;
        }
    </style>
</head>
<body>
    <h1>Todo List</h1>
    <textarea id="newTask" placeholder="Add a new task..."></textarea>
    <button onclick="addTask()">Add</button>
    <ul id="taskList"></ul>

    <script>
        let tasks = <?php echo json_encode($tasks); ?>;
        let draggedTask;
        let fullScreenTextarea;
        const project = '<?php echo $project; ?>';

        function addTask() {
            let taskInput = document.getElementById("newTask");
            let taskText = taskInput.value.trim();
            if (taskText !== "") {
                let task = {
                    text: nl2br(taskText),
                    completed: false,
                    editing: false,
                    project: project,
                    id: Date.now() // Temporary unique ID
                };
                tasks.unshift(task); // Add new task at the top
                taskInput.value = "";
                renderTasks();
                saveTasksToFile();
            }
        }

        function toggleTaskCompletion(taskId) {
            let xhr = new XMLHttpRequest();
            xhr.open("POST", "", true);
            xhr.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");
            xhr.onreadystatechange = function() {
                if (xhr.readyState === 4 && xhr.status === 200) {
                    tasks = JSON.parse(xhr.responseText);
                    renderTasks();
                }
            };

            let action = tasks.find(task => task.id === taskId).completed ? 'undo' : 'complete';
            xhr.send("action=" + action + "&taskId=" + taskId);
        }

        function editTask(taskId) {
            let task = tasks.find(task => task.id === taskId);
            task.editing = true;
            renderTasks();
            fullScreenTextarea = document.createElement("textarea");
            fullScreenTextarea.classList.add("full-screen");
            fullScreenTextarea.value = task.text.replace(/<br \/>/g, "\n").replace(/~([^~]+)~/g, function(match, p1) {
                return "<strike>" + p1 + "</strike>";
            });
            fullScreenTextarea.addEventListener("blur", function() {
                updateTask(taskId, fullScreenTextarea.value);
                fullScreenTextarea.remove();
            });
            document.body.appendChild(fullScreenTextarea);
            fullScreenTextarea.focus();
        }

        function updateTask(taskId, newText) {
            let task = tasks.find(task => task.id === taskId);
            task.text = nl2br(newText.trim()).replace(/~([^~]+)~/g, "<strike>$1</strike>");
            task.editing = false;
            renderTasks();
            saveTasksToFile();
        }

        function removeTask(taskId) {
            let index = tasks.findIndex(task => task.id === taskId);
            tasks.splice(index, 1);
            renderTasks();
            saveTasksToFile();
        }

        function renderTasks() {
            let taskList = document.getElementById("taskList");
            taskList.innerHTML = "";
            let filteredTasks = project === 'default' ? tasks : tasks.filter(task => task.project === project);
            let completedTasks = filteredTasks.filter(task => task.completed);
            let uncompletedTasks = filteredTasks.filter(task => !task.completed);
            for (let i = 0; i < uncompletedTasks.length; i++) {
                let task = uncompletedTasks[i];
                let li = createTaskElement(task, i, uncompletedTasks);
                taskList.appendChild(li);
            }
            for (let i = 0; i < completedTasks.length; i++) {
                let task = completedTasks[i];
                let li = createTaskElement(task, i, completedTasks);
                taskList.appendChild(li);
            }
        }

        function createTaskElement(task, index, taskArray) {
            let li = document.createElement("li");
            li.draggable = true;
            li.addEventListener("dragstart", dragStartHandler);
            li.addEventListener("dragover", dragOverHandler);
            li.addEventListener("drop", dropHandler);
            let taskText;
            if (task.editing) {
                let editInput = document.createElement("textarea");
                editInput.value = task.text.replace(/<br \/>/g, "\n").replace(/<strike>([^<]+)<\/strike>/g, function(match, p1) {
                    return "~" + p1 + "~";
                });
                editInput.addEventListener("blur", function() {
                    updateTask(task.id, editInput.value);
                });
                li.appendChild(editInput);
            } else {
                taskText = document.createElement("span");
                taskText.innerHTML = task.text;
                li.appendChild(taskText);
            }
            let buttonContainer = document.createElement("div");
            buttonContainer.classList.add("button-container");
            if (task.completed) {
                li.classList.add("completed");
            }
            let completeButton = document.createElement("button");
            completeButton.textContent = task.completed ? "Undo" : "Complete";
            completeButton.addEventListener("click", function() {
                toggleTaskCompletion(task.id);
            });
            buttonContainer.appendChild(completeButton);
            let editButton = document.createElement("button");
            editButton.textContent = task.editing ? "Save" : "Edit";
            editButton.addEventListener("click", function() {
                if (task.editing) {
                    let editInput = li.querySelector("textarea");
                    updateTask(task.id, editInput.value);
                } else {
                    editTask(task.id);
                }
            });
            buttonContainer.appendChild(editButton);
            let removeButton = document.createElement("button");
            removeButton.textContent = "Remove";
            removeButton.addEventListener("click", function() {
                removeTask(task.id);
            });
            buttonContainer.appendChild(removeButton);
            li.appendChild(buttonContainer);
            let moveIcon = document.createElement("span");
            moveIcon.classList.add("move-icon");
            moveIcon.textContent = "?";
            li.appendChild(moveIcon);
            return li;
        }

        function nl2br(str) {
            return str.replace(/\n/g, "<br />");
        }

        function saveTasksToFile() {
            let tasksData = JSON.stringify(tasks);
            let xhr = new XMLHttpRequest();
            xhr.open("POST", "", true);
            xhr.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");
            xhr.onreadystatechange = function() {
                if (xhr.readyState === 4 && xhr.status === 200) {
                    console.log("Tasks saved successfully.");
                }
            };
            xhr.send("tasks=" + encodeURIComponent(tasksData));
        }

        function dragStartHandler(e) {
            draggedTask = e.target;
            e.dataTransfer.effectAllowed = "move";
            e.dataTransfer.setData("text/html", null);
        }

        function dragOverHandler(e) {
            e.preventDefault();
            e.dataTransfer.dropEffect = "move";
        }

        function dropHandler(e) {
            e.preventDefault();
            let target = e.target.closest("li");
            let targetIndex = Array.from(target.parentNode.children).indexOf(target);
            let draggedIndex = Array.from(target.parentNode.children).indexOf(draggedTask);
            if (targetIndex !== draggedIndex) {
                tasks.splice(targetIndex, 0, tasks.splice(draggedIndex, 1)[0]);
                renderTasks();
                saveTasksToFile();
            }
        }

        renderTasks();
    </script>
</body>
</html>
