<?php
class ElasticsearchService {
    private $baseUrl;
    private $index;

    public function __construct() {
        $this->baseUrl = ES_URL;
        $this->index = ES_INDEX;
    }

    /**
     * Make HTTP request to Elasticsearch
     */
    private function request($endpoint, $method = 'GET', $data = null) {
        $url = "{$this->baseUrl}/{$this->index}/{$endpoint}";
        
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        
        if ($data) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        }

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            throw new Exception("Elasticsearch request failed: $error");
        }

        if ($httpCode >= 400) {
            throw new Exception("Elasticsearch error: $response");
        }

        return json_decode($response, true);
    }

    /**
     * Search articles with query
     */
    public function searchArticles($query, $page = 0) {
        $size = PAGE_SIZE;
        $from = $page * $size;
        
        // Normalize German characters
        $normalizedQuery = $this->normalizeGerman($query);
        
        $searchBody = [
            'query' => [
                'bool' => [
                    'must' => [
                        [
                            'bool' => [
                                'should' => [
                                    [
                                        'multi_match' => [
                                            'query' => str_replace('ß', 'ss', $query),
                                            'fields' => ['title^2', 'summary'],
                                            'type' => 'phrase_prefix',
                                            'minimum_should_match' => '90%'
                                        ]
                                    ],
                                    [
                                        'multi_match' => [
                                            'query' => str_replace('ss', 'ß', $query),
                                            'fields' => ['title^2', 'summary'],
                                            'type' => 'phrase_prefix',
                                            'minimum_should_match' => '90%'
                                        ]
                                    ],
                                    [
                                        'multi_match' => [
                                            'query' => $normalizedQuery,
                                            'fields' => ['title^2', 'summary'],
                                            'type' => 'cross_fields',
                                            'operator' => 'and'
                                        ]
                                    ]
                                ],
                                'minimum_should_match' => 1
                            ]
                        ]
                    ]
                ]
            ],
            'size' => $size,
            'from' => $from,
            'sort' => [['_doc' => 'desc']],
            'track_total_hits' => true,
            'highlight' => [
                'fields' => [
                    'summary' => new stdClass(),
                    'title' => new stdClass()
                ]
            ]
        ];

        return $this->request('_search', 'POST', $searchBody);
    }

    /**
     * Get categories with aggregations
     */
    public function getCategories() {
        $body = [
            'size' => 0,
            'aggs' => [
                'categories' => [
                    'terms' => [
                        'field' => 'category_label.keyword',
                        'size' => 100
                    ],
                    'aggs' => [
                        'latest_articles' => [
                            'top_hits' => [
                                'size' => 3,
                                'sort' => [['_doc' => 'desc']]
                            ]
                        ]
                    ]
                ]
            ],
            'query' => [
                'range' => [
                    'unixts_published' => [
                        'gte' => '0'
                    ]
                ]
            ]
        ];

        return $this->request('_search', 'POST', $body);
    }

    /**
     * Get domains by category
     */
    public function getDomainsByCategory($category) {
        $body = [
            'size' => 0,
            'query' => [
                'bool' => [
                    'must' => [
                        [
                            'term' => [
                                'category_label.keyword' => $category
                            ]
                        ],
                        [
                            'exists' => [
                                'field' => 'origin_stream_id.keyword'
                            ]
                        ]
                    ]
                ]
            ],
            'aggs' => [
                'domains' => [
                    'terms' => [
                        'field' => 'origin_stream_id.keyword',
                        'size' => 100,
                        'order' => ['_count' => 'desc']
                    ],
                    'aggs' => [
                        'latest_doc' => [
                            'max' => [
                                'field' => '_doc'
                            ]
                        ],
                        'latest_articles' => [
                            'top_hits' => [
                                'size' => 3,
                                'sort' => [['_doc' => 'desc']]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        return $this->request('_search', 'POST', $body);
    }

    /**
     * Get articles by domain
     */
    public function getArticlesByDomain($domain, $page = 0) {
        $size = PAGE_SIZE;
        $from = $page * $size;

        $body = [
            'size' => $size,
            'from' => $from,
            'query' => [
                'bool' => [
                    'must' => [
                        [
                            'term' => [
                                'origin_stream_id.keyword' => $domain
                            ]
                        ]
                    ]
                ]
            ],
            'collapse' => [
                'field' => 'title.keyword'
            ],
            'sort' => [['_doc' => 'desc']],
            'track_total_hits' => true
        ];

        return $this->request('_search', 'POST', $body);
    }

    /**
     * Normalize German characters
     */
    private function normalizeGerman($text) {
        $text = iconv('UTF-8', 'ASCII//TRANSLIT', $text);
        $text = str_replace(['-', 'ä', 'Ä', 'ö', 'Ö', 'ü', 'Ü', 'ß'], 
                          ['', 'a', 'a', 'o', 'o', 'u', 'u', 'ss'], 
                          $text);
        return $text;
    }
}