<?php
require_once 'config.php';
require_once 'ElasticsearchService.php';

try {
    $service = new ElasticsearchService();
    $response = ['error' => 'Invalid endpoint'];
    $statusCode = 400;

    // Parse the request path, removing the base path (/reader_api/v1/)
    $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
    $basePath = '/reader_api/v1/';
    
    // Remove base path from the request path
    if (strpos($path, $basePath) === 0) {
        $path = substr($path, strlen($basePath));
    }
    
    // Get the endpoint name
    $endpoint = trim($path, '/');
    if (empty($endpoint)) {
        $endpoint = 'index';
    }

    // Get query parameters
    $params = $_GET;
    
    // Log request details for debugging
    error_log("Path: " . $path);
    error_log("Endpoint: " . $endpoint);
    error_log("Params: " . json_encode($params));
    
    switch ($endpoint) {
        case 'search':
            if (!empty($params['q'])) {
                $page = isset($params['page']) ? intval($params['page']) : 0;
                $response = $service->searchArticles($params['q'], $page);
                $statusCode = 200;
            } else {
                $response = ['error' => 'Search query is required'];
                $statusCode = 400;
            }
            break;

        case 'categories':
            $response = $service->getCategories();
            $statusCode = 200;
            break;

        case 'domains':
            if (!empty($params['category'])) {
                $response = $service->getDomainsByCategory($params['category']);
                $statusCode = 200;
            } else {
                $response = ['error' => 'Category parameter is required'];
                $statusCode = 400;
            }
            break;

        case 'articles':
            if (!empty($params['domain'])) {
                $page = isset($params['page']) ? intval($params['page']) : 0;
                $response = $service->getArticlesByDomain($params['domain'], $page);
                $statusCode = 200;
            } else {
                $response = ['error' => 'Domain parameter is required'];
                $statusCode = 400;
            }
            break;

        case 'index':
            // Return API documentation for root endpoint
            $response = [
                'status' => 'API is running',
                'endpoints' => [
                    'GET /reader_api/v1/search' => [
                        'description' => 'Search articles',
                        'parameters' => [
                            'q' => 'Search query (required)',
                            'page' => 'Page number (optional, default: 0)'
                        ]
                    ],
                    'GET /reader_api/v1/categories' => [
                        'description' => 'Get all categories with latest articles'
                    ],
                    'GET /reader_api/v1/domains' => [
                        'description' => 'Get domains by category',
                        'parameters' => [
                            'category' => 'Category name (required)'
                        ]
                    ],
                    'GET /reader_api/v1/articles' => [
                        'description' => 'Get articles by domain',
                        'parameters' => [
                            'domain' => 'Domain name (required)',
                            'page' => 'Page number (optional, default: 0)'
                        ]
                    ]
                ]
            ];
            $statusCode = 200;
            break;

        default:
            $response = ['error' => 'Endpoint not found'];
            $statusCode = 404;
    }

    http_response_code($statusCode);
    echo json_encode($response);

} catch (Exception $e) {
    error_log("API Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'error' => 'Internal server error',
        'message' => $e->getMessage()
    ]);
}